#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "keyb.h"
#include "files.h"
#include "general.h"

#ifdef __linux__
static char *home = NULL;
static int homelen = 0;

int ISDIR(struct stat *stat)
{
    return S_ISDIR(stat->st_mode);
}

time_t *ST_MTIME(struct stat *stat)
{
    return &stat->st_mtime;
}

off_t ST_SIZE(struct stat *stat)
{
    return stat->st_size;
}

char *GetHome(char *fname, char *out)
{
    char *strp,prev;

    if (home == NULL)
    {
        home = getenv("HOME");
        homelen = strlen(home);
    }

    strp = out; prev = '\0';
    while (*fname != '\0')
    {
        if (*fname == '~' && (*(fname+1) == '\0' || *(fname+1) == SLASH) && (prev == '\0' || prev == SLASH))
        {
            if (home == NULL) return NULL;
            memcpy(strp,home,homelen);
            strp += homelen;
        }
        else
            *strp++ = *fname;

        prev = *fname++;
    }
    *strp = '\0';

    return out;
}
#endif

char *_fgets(char *buf, size_t n, FILE *fp)
{
    char *strp;

    if (fgets(buf,n,fp) == NULL) return NULL;
    strp = buf;
    while (*strp != '\0')
    {
        if (*strp == '\r' || *strp == '\n')
        {
            *strp = '\0';
            break;
        }
        strp++;
    }
    while (strp > buf && *(strp-1) == ' ') strp--; *strp = '\0';

    return buf;
}

char exists(char *fname)
{
    DIR *dirp;
    char *name;

    fname = strdup(fname);
    name = strrchr(fname, SLASH);
    if (name == NULL)
    {
        name = fname;
        dirp = OpenDir(".");
    }
    else
    {
        *name++ = '\0';
        dirp = OpenDir(fname);
    }

    for (;;)
    {
        struct dirent *d;
        if (dirp == NULL) break;

        d = readdir(dirp);
        if (d == NULL) break;

        if (compare_filenames(d->d_name, name))
        {
            free(fname);
            return 1;
        }
    }

    free(fname);
    return 0;
}

char *findfile(char *fname, char *out)
{
    DIR *dirp;
    char *name;

    fname = strdup(fname);
    name = strrchr(fname, SLASH);
    if (name == NULL)
    {
        name = fname;
        dirp = OpenDir(".");
    }
    else
    {
        *name++ = '\0';
        dirp = OpenDir(fname);
    }

    for (;;)
    {
        struct dirent *d;
        if (dirp == NULL) break;

        d = readdir(dirp);
        if (d == NULL) break;

        if (compare_filenames(d->d_name, name))
        {
            strcpy(out, d->d_name);
            free(fname);
            return out;
        }
    }

    free(fname);
    return NULL;
}

char isdir(char *path)
{
    struct stat *statbuf;
    int ret;

    if (*path == '\0') return 1;
    path = strdup(path);
    if (path[strlen(path)-1] == SLASH) path[strlen(path)-1] = '\0';
    ret = FileStat(path, &statbuf) == 0 && S_ISDIR(statbuf->st_mode);
    free(statbuf);
    free(path);
    return ret;
}

char make_space(FILE *Fhandle, long pos, long size, long fsize)
{
    char tmpbuf[128];
    char *buffer;

    unsigned bufsize;
    unsigned long movesz,lpos;

    bufsize = 16384;
    movesz = fsize - pos;
    if (bufsize > movesz) bufsize = movesz;
    if (bufsize == 0) return 0;

    if ((bufsize <= sizeof(tmpbuf)) || ((buffer = (char *) malloc(bufsize)) == NULL))
    {
        bufsize = sizeof(tmpbuf);
        buffer = tmpbuf;
    }

    for (lpos=1; lpos<=movesz/bufsize; lpos++)
    {
        fseek(Fhandle,fsize-bufsize*lpos,SEEK_SET);
        if (fread(buffer,bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
        fseek(Fhandle,fsize-bufsize*lpos+size,SEEK_SET);
        if (fwrite(buffer,bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
    }

    if (movesz % bufsize > 0) {
        fseek(Fhandle,pos,SEEK_SET);
        if (fread(buffer,movesz % bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
        fseek(Fhandle,pos+size,SEEK_SET);
        if (fwrite(buffer,movesz % bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
    }

    if (buffer != tmpbuf) free(buffer);
    return 1;
}

char delete_space(FILE *Fhandle, long pos, long size, long fsize)
{
    char tmpbuf[128];
    char *buffer;

    unsigned bufsize;
    unsigned long movesz,lpos;

    bufsize = 16384;
    movesz = fsize - pos - size;
    if (bufsize > movesz) bufsize = movesz;
    if (bufsize == 0) return 0;

    if ((bufsize <= sizeof(tmpbuf)) || ((buffer = (char *) malloc(bufsize)) == NULL))
    {
        bufsize = sizeof(tmpbuf);
        buffer = tmpbuf;
    }

    for (lpos=1; lpos<=movesz/bufsize; lpos++)
    {
        fseek(Fhandle,pos+size+(lpos-1)*bufsize,SEEK_SET);
        fread(buffer,bufsize,1,Fhandle);
        fseek(Fhandle,pos+(lpos-1)*bufsize,SEEK_SET);
        if (fwrite(buffer,bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
    }

    if (movesz % bufsize > 0)
    {
        fseek(Fhandle,pos+size+(movesz/bufsize)*bufsize,SEEK_SET);
        fread(buffer,movesz % bufsize,1,Fhandle);
        fseek(Fhandle,pos+(movesz/bufsize)*bufsize,SEEK_SET);
        if (fwrite(buffer,movesz % bufsize,1,Fhandle) != 1)
        {
            if (buffer != tmpbuf) free(buffer);
            return 0;
        }
    }

    if (buffer != tmpbuf) free(buffer);
    return 1;
}

#ifdef __linux__
char mkpath(char *_path)
#else
char mkpath(char *path)
#endif
{
    char str[256];
#ifdef __linux__
    char path[1024];
#endif
#ifndef __NO_DRIVES__
    unsigned drive;
#endif
    int pos,slen,spos;
    int start;

#ifdef __linux__
    if (GetHome(_path,path) == NULL) return 0;
#endif
    slen = strlen(path);

    if (path[slen-1] == SLASH) slen--;

    start = 0;
#ifndef __NO_DRIVES__
    if (path[1] == ':')
    {
        drive = toupper(path[0]);
        start = 2;
    }
    else
    {
        _dos_getdrive(&drive); drive += 'A'-1;
    }

    if (path[start] == SLASH)
    {
        str[0] = (char) drive;
        str[1] = ':';
        str[2] = SLASH;
        str[3] = '\0';
        start++;
    } else str[0] = '\0';
#else
    str[0] = '\0';
#endif

    spos = strlen(str);
    do
    {
        for (pos=start; pos<slen; pos++)
        {
            if (path[pos] == SLASH) break;

            str[spos] = path[pos];
            spos++;
        }
        str[spos] = 0;

        start = pos+1;
        if (!isdir(str))
        {
#ifdef __linux__
            if (mkdir(str,0755) != 0) return 0;
#else
            if (mkdir(str) != 0) return 0;
#endif
        }

        str[spos] = SLASH; str[spos+1] = 0; spos++;
    } while (slen >= start);

    return 1;
}

char copyfile(char *src, char *dest)
{
    FILE *Fsrc,*Fdst;
    char *buf;
    size_t readed;

    if ((Fsrc = FileOpen(src,"rb")) == NULL) return 0;
    if ((Fdst = FileOpen(dest,"w+b")) == NULL) { fclose(Fsrc); return 0; }
    buf = (char *) malloc(4096);

    do {
        readed = fread(buf,1,4096,Fsrc);
        fwrite(buf,1,readed,Fdst);
    } while (readed == 4096);

    free(buf);
    fclose(Fsrc); fclose(Fdst);

    return 1;
}

#ifdef __linux__

FILE *fopen_ign(char *fname, char *mode)
{
    find_t SR;
    char str[1024],*pstr;

    strcpy(str, fname);
    if (_dos_findfirst(str,0,&SR) != 0) return NULL;
    pstr = strrchr(fname,SLASH);
    if (pstr == NULL)
    {
        strcpy(str,SR.name);
    }
    else
    {
        *pstr = '\0';
        sprintf(str, "%s/%s", fname, SR.name);
        *pstr = SLASH;
    }
    _dos_findclose(&SR);
    return FileOpen(str, mode);
}

FILE *FileOpen(char *fname, char *mode)
{
    char name[1024];

    if (GetHome(fname,name) == NULL) return NULL;
    return fopen(name,mode);
}

DIR *OpenDir(char *fname)
{
    char name[1024];

    if (GetHome(fname,name) == NULL) return NULL;
    return opendir(name);
}

int FileStat(char *fname, struct stat **buf)
{
    char name[1024];

    *buf = (struct stat *) malloc(sizeof(struct stat));
    if (GetHome(fname,name) == NULL) return 1;
    return stat(name,*buf);
}

int FileFStat(int h, struct stat **buf)
{
    *buf = (struct stat *) malloc(sizeof(struct stat));
    return fstat(h, *buf);
}

int FileRename(char *from, char *to)
{
    char name1[512],name2[512];

    if (GetHome(from,name1) == NULL) return 1;
    if (GetHome(to,name2) == NULL) return 1;
    return rename(name1,name2);
}

int FileRemove(char *fname)
{
    char name[1024];

    if (GetHome(fname,name) == NULL) return 1;
    return remove(name);
}

int ChangeDir(char *path)
{
    char name[1024];

    if (GetHome(path,name) == NULL) return 1;
    return chdir(name);
}

int FileTime(char *path, struct utimbuf *buf)
{
    char name[1024];

    if (GetHome(path, name) == NULL) return 1;
    return utime(name, buf);
}

unsigned _dos_findfirst(char *path, unsigned attributes, find_t *buffer)
{
#ifdef __linux__
    char path2[1024];
#endif
    char *pstr,*slash;
    int num,pos,point;
    struct stat statbuf;

#ifdef __linux__
    if (GetHome(path,path2) == NULL) return 0;
    slash = strrchr(path2,SLASH);
#else
    slash = strrchr(path,SLASH);
#endif

    if (slash != NULL)
    {
        *slash = '\0';
        buffer->attributes = attributes;
        strcpy(buffer->search,slash+1);
#ifdef __linux__
        buffer->dir = opendir(path2);
#else
        buffer->dir = opendir(path);
#endif
    }
    else
    {
#ifdef __linux__
        strcpy(buffer->search,path2);
#else
        strcpy(buffer->search,path);
#endif
        buffer->dir = opendir(".");
    }

    if (buffer->dir == NULL)
    {
        if (slash != NULL) *slash = SLASH;
        return 1;
    }

    for (;;)
    {
        buffer->direntp = readdir(buffer->dir);
        if (buffer->direntp == NULL)
        {
            closedir(buffer->dir);
            buffer->dir = NULL;
            if (slash != NULL) *slash = SLASH;
            return 1;
        }

        pstr = buffer->direntp->d_name;

        if (stat(pstr,&statbuf) == 0)
            if (S_ISDIR(statbuf.st_mode) && (attributes & _A_SUBDIR) == 0) continue;

        pos = 0; point = 0; num = 0;
        while (num > -1)
        {
            switch(buffer->search[pos])
            {
                case 0:
                    num = -1;
                    break;
                case '*':
                    if (point == 0) {
                        pstr = strrchr(buffer->direntp->d_name,'.');
                        if ((pstr == NULL) || (pstr < &buffer->direntp->d_name[num]))
                            pstr = buffer->direntp->d_name+strlen(buffer->direntp->d_name);
                        num = 0;
                    } else
                        num = -1;
                    break;
                case '.':
                    if (!((pstr[num] == '.') || (pstr[num] == 0)))
                        num = -2;
                    else
                        num++;
                    point = 1;
                    break;
                case '?':
                    if (pstr[num] != 0) num++;
                    break;
                default:
                    if (toupper(pstr[num]) != toupper(buffer->search[pos]))
                        num = -2;
                    else
                        num++;
                    break;
            }
            pos++;
        }

        if (num == -1)
        {
            strcpy(buffer->name,buffer->direntp->d_name);
            break;
        }
    }

    if (slash != NULL) *slash = SLASH;
    return 0;
}

unsigned _dos_findnext(find_t *buffer)
{
    char *pstr;
    int num,pos,point;
    struct stat statbuf;

    for (;;)
    {
        buffer->direntp = readdir(buffer->dir);
        if (buffer->direntp == NULL) return 1;

        pstr = buffer->direntp->d_name;
        if (stat(pstr,&statbuf) == 0)
            if (S_ISDIR(statbuf.st_mode) && (buffer->attributes & _A_SUBDIR) == 0) continue;

        pos = 0; point = 0; num = 0;
        while (num > -1)
        {
            switch (buffer->search[pos])
            {
                case 0:
                    num = -1;
                    break;
                case '*':
                    if (point == 0) {
                        pstr = strrchr(buffer->direntp->d_name,'.');
                        if ((pstr == NULL) || (pstr < &buffer->direntp->d_name[num]))
                            pstr = &buffer->direntp->d_name[strlen(buffer->direntp->d_name)];
                        num = 0;
                    } else
                        num = -1;
                    break;
                case '.':
                    if (!((pstr[num] == '.') || (pstr[num] == 0)))
                        num = -2;
                    else
                        num++;
                    point = 1;
                    break;
                case '?':
                    if (pstr[num] != 0) num++;
                    break;
                default:
                    if (toupper(pstr[num]) != toupper(buffer->search[pos]))
                        num = -2;
                    else
                        num++;
                    break;
            }
            pos++;
        }

    }
    return 0;
}

unsigned _dos_findclose(find_t *buffer)
{
    if (buffer->dir != NULL)
        closedir(buffer->dir);
    return 0;
}
#endif

char cmpfile(char *file1, char *file2)
{
    FILE *F1,*F2;
    char *buf,*buf2,ok;
    int num;
    size_t readed;

    if ((F1 = FileOpen(file1,"rb")) == NULL) return 1;
    if ((F2 = FileOpen(file2,"rb")) == NULL) { fclose(F1); return 1; }

    buf = (char *) malloc(2048);
    buf2 = (char *) malloc(2048);

    if (buf == NULL || buf2 == NULL)
    {
        fclose(F1);
        fclose(F2);
        return 0;
    }

    ok = 1;
    for (;;)
    {
        readed = fread(buf,1,2048,F1);
        if (fread(buf2,1,2048,F2) == readed)
        {
            if (readed == 0)
            {
                ok = 1;
                break;
            }
            for (num=0; num<2048; num++)
            {
                if (buf[num] != buf[num])
                {
                    ok = 0;
                    break;
                }
            }
            if (!ok) break;
        }
        else
        {
            ok = 0;
            break;
        }
    }

    free(buf);
    free(buf2);

    fclose(F1); fclose(F2);
    return ok;
}

/* Compare if file matches search string */
int compare_filenames(char *fname, char *wildcard)
{
    char *strp,*strp2,*wptr;
    int quit,wild;
    
    if (wildcard[0] == '\0') return 1;
    
    while (wildcard != NULL)
    {
        wptr = strchr(wildcard,' ');
        if (wptr != NULL) *wptr = '\0';
        
        strp = fname;
        quit = 0;
        wild = 0;
        while (!quit)
        {
            switch(*wildcard)
            {
                case '\0':
                    if (wptr != NULL) *wptr = ' ';
                    if ((*strp != '\0') && (!wild)) return 0;
                    return 1;
                case '*':
                    strp2 = strrchr(fname,'.');
                    if ((strp2 == NULL) || (strp2 < strp))
                    {
                        strp = &fname[strlen(fname)];
                    }
                    else
                    {
                        strp = strp2;
                    }
                    wild = 1;
                    break;
                case '.':
                    if ((*strp != '.') && (*strp != '\0'))
                    {
                        quit = 1;
                        break;
                    }
                case '?':
                    if (*strp != '\0') strp++;
                    wild = 0;
                    break;
                default:
                    if (toupper(*strp) != toupper(*wildcard))
                    {
                        quit = 1;
                        break;
                    }
                    strp++;
                    wild = 0;
                    break;
            }
            wildcard++;
        }
        if (wptr != NULL)
        {
            *wptr = ' ';
            wildcard = wptr+1;
        }
        else
        {
            wildcard = NULL;
        }
    }
    
    return 0;
}

#ifdef __linux__
char *search_ign_file(char *fname)
{
    find_t SR;
    char *pstr;

    if (_dos_findfirst(fname,0,&SR) != 0) return NULL;
    pstr = strrchr(fname,SLASH);
    if (pstr == NULL)
    {
        strcpy(fname,SR.name);
    }
    else
    {
        *(pstr+1) = '\0';
        strcat(fname,SR.name);
    }
    _dos_findclose(&SR);
    return fname;
}
#endif
