/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* fsed.c - Full screen editor */

/* Todo: add_character() should not be recursive */

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "screen.h"
#include "files.h"
#include "keyb.h"
#include "vars.h"
#include "scrsaver.h"

#define QUOTE_MAX 7

typedef struct ELINE_REC
{
    struct ELINE_REC *next;
    struct ELINE_REC *prev;

    unsigned char len;
    char joined; /* Next line belongs to same chapter */
    char text[81];
}
ELINE_REC;

typedef struct ULINE_REC
{
    struct ULINE_REC *next;
    struct ULINE_REC *prev;

    char text[81];
}
ULINE_REC;

static ELINE_REC *first_line, *line;
static int xpos, ypos;
static int linecol, oldcol;
static unsigned lines, upy;

static int header_lines;

static int cur_color;

int line_color(ELINE_REC *line)
{
    char *strp;
    int num;

    if (strncmp(line->text, "... ", 4) == 0)
    {
        cur_color = color[col_editor_tagline];
        return 1;
    }
    else
    {
        strp = line->text;
        for (num = 0; num < QUOTE_MAX && *strp != '\0'; num++)
        {
            if (*strp == '"' || *strp == '\'' || *strp == '<') break;
            if (*strp == '>')
            {
                num = 100;
                break;
            }
            strp++;
        }
        if (num == 100)
        {
            cur_color = color[col_editor_quote];
            return 2;
        }
        else
        {
            cur_color = color[col_editor_text];
            return 3;
        }
    }
}

void add_character(unsigned char ch)
{
    ELINE_REC *tmpline;
    int oldx, bpos;
    char *strp,buffer[80];

    while (line->len == 79)
    {
        strp = line->text+78;
        while (*strp == ' ' && strp > line->text) strp--;
        while (*strp != ' ' && strp > line->text) strp--;
        if (strp > line->text)
        {
            strcpy(buffer,strp+1);
            line->len = (unsigned char) ((strp-line->text)+1);
            bpos = strlen(buffer)-(xpos-line->len);
        }
        else
        {
            strcpy(buffer,line->text+78);
            line->len--;
            bpos = 0;
        }

        if (line->next == NULL || line->joined == 0)
        {
            /* Make new line */
            if (line->next != NULL && ypos < (int) scrsize)
            {
                scroll_down(1, ypos+1, scrwidth, scrsize);
                tbar(1, ypos+1, scrwidth, ypos+1, cur_color);
                //output("@CR@@INSLINE@");
            }
            line->joined = 1;
            tmpline = (ELINE_REC *) malloc(sizeof(ELINE_REC));
            tmpline->prev = line;
            tmpline->next = line->next;
            if (line->next != NULL) line->next->prev = tmpline;
            line->next = tmpline;
            line = tmpline;

            line->len = 0;
            line->joined = 0;
            line->text[0] = '\0';
            lines++;
        }
        else
        {
            line = line->next;
        }

        oldx = xpos;
        xpos = 0; ypos++;
        strp = buffer;
        //if (ypos <= scrsize) output("@GOTO:1,%d@",ypos);
        while (*strp != '\0') add_character(*strp++);

        ypos--; line = line->prev;
        line->text[line->len] = '\0';
        oldcol = line_color(line);
        linecol = 1;
        if (ypos <= (int) scrsize)
        {
            tbar(line->len+1, ypos, scrwidth, ypos, cur_color);
            //output("@GOTO:%d,%d@@CLREOL@",line->len+1,ypos);
        }

        if (oldx >= line->len)
        {
            line = line->next;
            if (ypos < (int) scrsize)
            {
                ypos++;
            }
            else
            {
                upy++;
                scroll_down(1, 2, scrwidth, scrsize);
                cwritexy(1, ypos, line->text, cur_color);
                //output("@SCROLL_UP:2@@GOTO:1,%d@%s",ypos,line->text);
            }
            xpos -= bpos;
        }
        else
        {
            xpos = oldx;
        }
        //if (ypos <= scrsize) output("@GOTO:%d,%d@",xpos+1,ypos);
    }

    if (!linecol) oldx = line_color(line); else oldx = oldcol;
    memmove(line->text+xpos+1,line->text+xpos,line->len-xpos+1);
    line->text[xpos] = ch; line->len++; xpos++;

    if (ypos <= (int) scrsize)
    {
        if (!linecol || xpos <= QUOTE_MAX)
        {
            oldcol = line_color(line);
            linecol = 1;
            if (oldx != oldcol)
            {
                cwritexy(1, ypos, line->text, cur_color);
                /*outchr('\r');
                outtext(line->text);
                output("@GOTO:%d,%d@", xpos+1, ypos);*/
                return;
            }
        }

        cwritexy(xpos, ypos, line->text+xpos-1, cur_color);
        //output("@GOTO:%d,%d@",xpos+1,ypos);
    }
}

void add_string(char *text)
{
    while (*text != '\0') add_character(*text++);
}

void refresh_screen(void)
{
    ELINE_REC *tmpline;
    int num;

    tbar(1,header_lines+1,scrwidth,scrsize,color[col_editor_text]);
    tmpline = first_line; num = upy;
    while (num > 0 && tmpline != NULL)
    {
        tmpline = tmpline->next;
        num--;
    }

    if (tmpline != NULL)
    {
        for (num = header_lines+1; num <=(int) scrsize; num++)
        {
            oldcol = line_color(tmpline);
            cwritexy(1, num, tmpline->text, cur_color);
            //outtext(tmpline->text);
            //if (num < scrsize) output("\r\n");
            tmpline = tmpline->next;
            if (tmpline == NULL) break;
        }
    }
    gotoxy(xpos+1, ypos);
    //output("@GOTO:%d,%d@",xpos+1,ypos);
    linecol = 0;
}

int read_line(FILE *F, char *str)
{
    int readed, pos, space, joined;

    readed = fread(str, 1, 81, F);
    if (readed == 0) return 2;

    space = -1;
    for (pos = 0; str[pos] != '\r' && str[pos] != '\n' && pos < 80 && pos < readed; pos++)
    {
        if (str[pos] == ' ') space = pos;
    }
    if (pos == 80)
    {
        if (space > 0)
        {
            str[space] = '\0';
            pos = space+1;
        }
        else
            str[pos] = '\0';
        joined = 1;
    }
    else
    {
        if (str[pos] == '\r' && str[pos+1] == '\n')
        {
            str[pos] = '\0';
            pos += 2;
        }
        else
            str[pos++] = '\0';
        joined = 0;
    }
    fseek(F, -readed+pos, SEEK_CUR);
    return joined;
}

/* Read file to message */
void read_file(char *fname)
{
    char str[83];
    ELINE_REC *tmpline;
    FILE *F;
    int joined;

    F = fopen(fname, "rb");
    if (F == NULL) return;

    for (;;)
    {
        joined = read_line(F, str);
        if (joined == 2) break;
        line->joined = joined;
        strcpy(line->text, str);
        line->len = (unsigned char) strlen(line->text);

        /* Make new line */
        tmpline = (ELINE_REC *) malloc(sizeof(ELINE_REC));
        line->next = tmpline;
        tmpline->prev = line;
        tmpline->next = NULL;
        line = tmpline;

        line->len = 0;
        line->joined = 0;
        line->text[0] = '\0';
        lines++;
    }

    if (fmt != format_bw)
    {
        line = first_line;
        while (line != NULL)
        {
            if (line->joined == 0 && line->next != NULL && line->next->len != 0 && line->len > 50)
                line->joined = 1;
            line = line->next;
        }
    }

    line = first_line;
    fclose(F);
}

/* Write message to file */
void save_file(char *fname)
{
    FILE *F;

    F = fopen(fname, "w+b");
    if (F == NULL) return;

    line = first_line;
    while (line != NULL)
    {
        if (line->len > 0) fwrite(line->text, line->len, 1, F);
        if (!line->joined || fmt != format_bw)
            fwrite("\r", 1, 1, F);
        else
            if (line->text[line->len-1] != ' ') fwrite(" ", 1, 1, F);
        line = line->next;
    }

    fclose(F);
}

int fs_editor(char *fname)
{
    char ch, quit, buffer[81];
    int num, oldx, oldy;
    ELINE_REC *tmpline;
    ULINE_REC *ufirst, *ulast, *utmp;
    int ulines;

    tbar(1,1,scrwidth,1,color[col_title]);
    cmiddle(1, "Ctrl-Y = Remove line, Ctrl-Z = Save, Ctrl-A = Abort", color[col_title]);
    draw_clock();
    header_lines = 1;

    first_line = (ELINE_REC *) malloc(sizeof(ELINE_REC));
    line = first_line;
    line->len = 0;
    line->joined = 0;
    line->next = NULL;
    line->prev = NULL;
    line->text[0] = '\0';

    ufirst = NULL; ulast = NULL; ulines = 0;

    xpos = 0; ypos = header_lines+1; upy = 0; lines = 1;
    read_file(fname);
    refresh_screen();

    quit = 0;
    while (!quit)
    {
        if (sk_kbhit())
        {
            ch = (char) sk_getch();
            switch (ch)
            {
                case 0:
                    ch = (char) sk_getch();
                    switch (ch)
                    {
                        case 'K':
                            /* Left */
                            if (xpos > 0)
                            {
                                xpos--;
                                gotoxy(xpos+1, ypos);
                                //output("@GOTO:%d,%d@",xpos+1,ypos);
                            }
                            break;
                        case 'M':
                            /* Right */
                            if (xpos < line->len)
                            {
                                xpos++;
                                gotoxy(xpos+1, ypos);
                                //output("@GOTO:%d,%d@",xpos+1,ypos);
                            }
                            break;
                        case 'H':
                            /* Up */
                            if ((int) (ypos+upy) > header_lines+1)
                            {
                                line = line->prev;
                                if (xpos > line->len) xpos = line->len;

                                if (ypos > header_lines+1)
                                {
                                    ypos--; linecol = 0;
                                    gotoxy(xpos+1, ypos);
                                    //output("@GOTO:%d,%d@",xpos+1,ypos);
                                }
                                else
                                {
                                    upy--;
                                    scroll_down(1, header_lines+1, scrwidth, scrsize);
                                    //output("@SCROLL_DOWN:%d@\r",header_lines+1);
                                    oldcol = line_color(line);
                                    linecol = 1;
                                    tbar(1, ypos, scrwidth, ypos, cur_color);
                                    cwritexy(1, ypos, line->text, cur_color);
                                    gotoxy(xpos+1, ypos);
                                    //output("@CLREOL@@GOTO:%d,%d@",xpos+1,ypos);
                                }
                            }
                            break;
                        case 'P':
                            /* Down */
                            if (ypos+upy-header_lines < lines)
                            {
                                line = line->next;
                                if (xpos > line->len) xpos = line->len;
                                if (ypos < (int) scrsize)
                                {
                                    ypos++; linecol = 0;
                                    gotoxy(xpos+1, ypos);
                                    //output("@GOTO:%d,%d@",xpos+1,ypos);
                                }
                                else
                                {
                                    upy++;
                                    scroll_up(1, header_lines+1, scrwidth, scrsize);
                                    //output("@SCROLL_UP:%d@@GOTO:%d,%d@\r",header_lines+1,xpos+1,ypos);
                                    oldcol = line_color(line);
                                    linecol = 1;
                                    tbar(1, ypos, scrwidth, ypos, cur_color);
                                    cwritexy(1, ypos, line->text, cur_color);
                                    gotoxy(xpos+1, ypos);
                                    //outtext(line->text);
                                    //output("@CLREOL@@GOTO:%d,%d@",xpos+1,ypos);
                                }
                            }
                            break;
                        case 'G':
                            /* Home */
                            if (xpos > 0)
                            {
                                xpos = 0;
                                //output("@GOTO:1,%d@",ypos);
                                gotoxy(1, ypos);
                            }
                            break;
                        case 'O':
                            /* End */
                            if (xpos < line->len)
                            {
                                xpos = line->len;
                                //output("@GOTO:%d,%d@",xpos+1,ypos);
                                gotoxy(xpos+1, ypos);
                            }
                            break;
                        case 'S':
                            /* Delete */
                            if (xpos < line->len)
                                xpos++;
                            else
                            {
                                if (line->next == NULL) break;
                                ypos++; xpos = 0;
                                line = line->next;
                            }
                            goto __backspace;
                    }
                    break;
                case 1:
                    /* Ctrl-A = Abort */
                    quit = 1;
                    break;
                case 13:
                    /* Enter */

                    /* Make new line */
                    tmpline = (ELINE_REC *) malloc(sizeof(ELINE_REC));
                    tmpline->prev = line;
                    tmpline->next = line->next;
                    if (line->next != NULL) line->next->prev = tmpline;
                    line->next = tmpline; line->joined = 0;

                    strcpy(tmpline->text, line->text+xpos);
                    tmpline->len = (unsigned char) strlen(tmpline->text);
                    line->text[xpos] = '\0'; line->len = (unsigned char) xpos;
                    tmpline->joined = line->joined; line = tmpline;
                    lines++;
                    tbar(xpos+1, ypos, scrwidth, ypos, cur_color);
                    xpos = 0;
                    //output("@CLREOL@");
                    oldcol = line_color(line);
                    linecol = 1;
                    if (ypos == (int) scrsize)
                    {
                        /* Scroll down */
                        scroll_up(1, header_lines+1, scrwidth, scrsize);
                        tbar(1, ypos, scrwidth, ypos, cur_color);
                        cwritexy(1, ypos, line->text, cur_color);
                        //output("@SCROLL_UP:%d@@GOTO:1,%d@@CLREOL@%s\r", header_lines+1, ypos, line->text);
                        upy++;
                    }
                    else
                    {
                        //output("@CR@@INSLINE@%s\r",line->text);
                        ypos++;
                        scroll_down(1, ypos, scrwidth, scrsize);
                        tbar(1, ypos, scrwidth, ypos, cur_color);
                        cwritexy(1, ypos, line->text, cur_color);
                    }
                    gotoxy(xpos+1, ypos);
                    break;
                case 8:
                __backspace:
                    /* Backspace */
                    if (xpos == 0)
                    {
                        if ((int) (ypos+upy) == header_lines+1) break;

                        if (ypos == header_lines+1)
                        {
                            scroll_down(1, header_lines+1, scrwidth, scrsize);
                            tbar(1, ypos, scrwidth, ypos+1, cur_color);
                            oldcol = line_color(line->prev);
                            cwritexy(1, ypos, line->prev->text, cur_color);
                            oldcol = line_color(line);
                            upy--;
                        }
                        else
                        {
                            if (ypos <= scrsize) tbar(1, ypos, scrwidth, ypos, cur_color);
                            ypos--;
                        }

                        strcpy(buffer, line->text);
                        line->len = 0; line->text[0] = '\0';
                        line = line->prev;
                        if (line->text[line->len-1] != ' ' && line->joined)
                        {
                            line->text[line->len++] = ' ';
                            line->text[line->len] = '\0';
                        }
                        xpos = line->len; line->joined = 1;
                        gotoxy(xpos+1, ypos);
                        oldx = xpos; oldy = ypos; tmpline = line; linecol = 0;
                        add_string(buffer); xpos = oldx; ypos = oldy;
                        line = tmpline; tmpline = line->next;
                        if (tmpline->len == 0)
                        {
                            /* Remove line */
                            line->joined = tmpline->joined;
                            line->next = tmpline->next;
                            if (tmpline->next != NULL) tmpline->next->prev = line;
                            free(tmpline); lines--;
                            scroll_up(1, ypos+1, scrwidth, scrsize);
                            tmpline = line;
                            num = ypos;
                            while (num < (int) scrsize && tmpline != NULL)
                            {
                                tmpline = tmpline->next;
                                num++;
                            }
                            tbar(1, scrsize, scrwidth, scrsize, cur_color);
                            if (tmpline != NULL) cwritexy(1, scrsize, tmpline->text, cur_color);
                        }
                    }
                    else
                    {
                        xpos--;
                        memmove(line->text+xpos,line->text+xpos+1,line->len-xpos+1);
                        line->len--;

                        if (!linecol || xpos <= QUOTE_MAX)
                        {
                            linecol = 1;
                            oldx = oldcol;
                            oldcol = line_color(line);
                            if (oldx != oldcol)
                            {
                                cwritexy(1, ypos, line->text, cur_color);
                                writechr(line->len+1, ypos, ' ', cur_color);
                                /*outchr('\r');
                                outtext(line->text);
                                output(" @GOTO:%d,%d@", xpos+1, ypos);*/
                                oldx = 0;
                            }
                            else
                                oldx = 1;
                        }
                        else
                            oldx = 1;

                        if (oldx)
                        {
                            cwritexy(xpos+1, ypos, line->text+xpos, cur_color);
                            writechr(line->len+1, ypos, ' ', cur_color);
                            //output(" @GOTO:%d,%d@", xpos+1, ypos);
                        }
                    }
                    gotoxy(xpos+1, ypos);
                    break;
                case 18:
                    /* Ctrl-R = Redraw */
                    refresh_screen();
                    break;
                case 21:
                    /* Ctrl-U = Undelete line */
                    if (ufirst == NULL) break;
                    tmpline = (ELINE_REC *) malloc(sizeof(ELINE_REC));
                    tmpline->joined = line->prev != NULL ? line->prev->joined : 0;
                    tmpline->next = line;
                    tmpline->prev = line->prev;
                    if (line->prev != NULL) line->prev->next = tmpline; else first_line = tmpline;
                    line->prev = tmpline;
                    tmpline->len = strlen(ufirst->text);
                    strcpy(tmpline->text, ufirst->text);
                    line = tmpline; lines++;
                    utmp = ufirst; ufirst = ufirst->next; ulines--; free(utmp);
                    if (ufirst != NULL) ufirst->prev = NULL;
                    scroll_down(1, ypos, scrwidth, scrsize);
                    oldcol = line_color(line);
                    linecol = 1;
                    tbar(1, ypos, scrwidth, ypos, cur_color);
                    cwritexy(1, ypos, line->text, cur_color);
                    break;
                case 25:
                    /* Ctrl-Y = Remove line */
                    if (ulines >= 100)
                    {
                        ulast = ulast->prev;
                        free(ulast->next);
                        ulast->next = NULL;
                    }
                    else
                        ulines++;

                    utmp = (ULINE_REC *) malloc(sizeof(ULINE_REC));
                    if (utmp == NULL)
                    {
                        ulines--;
                        break;
                    }

                    utmp->next = ufirst;
                    utmp->prev = NULL;
                    if (ufirst != NULL) ufirst->prev = utmp; else ulast = utmp;
                    ufirst = utmp;
                    strcpy(ufirst->text, line->text);

                    if (line->next == NULL)
                    {
                        /* Last line, just clear line */
                        xpos = 0; line->len = 0; line->text[0] = '\0';
                        tbar(1, ypos, scrwidth, ypos, cur_color);
                    }
                    else
                    {
                        /* Delete line */
                        tmpline = line;
                        line = line->next;

                        if (tmpline == first_line) first_line = line;
                        if (tmpline->prev != NULL) tmpline->prev->next = line;
                        line->prev = tmpline->prev;
                        free(tmpline); lines--;
                        scroll_up(1, ypos, scrwidth, scrsize);

                        /* Draw last line */
                        tmpline = line;
                        num = ypos;
                        while (num < (int) scrsize && tmpline != NULL)
                        {
                            tmpline = tmpline->next;
                            num++;
                        }
                        tbar(1, scrsize, scrwidth, scrsize, cur_color);
                        if (tmpline != NULL)
                        {
                            oldcol = line_color(tmpline);
                            linecol = 1;
                            cwritexy(1, scrsize, tmpline->text, cur_color);
                        }

                        if (xpos > line->len) xpos = line->len;
                        gotoxy(xpos+1, ypos);
                    }
                    break;
                //case 24:
                case 19:
                case 26:
                    /* Ctrl-X/Z = Save */
                    save_file(fname);
                    quit = 2;
                    break;
                default:
                    if ((unsigned char) ch >= 32) add_character(ch);
                    gotoxy(xpos+1, ypos);
                    break;
            }
        }
        else
        {
            give_timeslice();
        }
    }

    /* Release memory */
    while (first_line != NULL)
    {
        line = first_line->next;
        free(first_line);
        first_line = line;
    }
    while (ufirst != NULL)
    {
        utmp = ufirst->next;
        free(ufirst);
        ufirst = utmp;
    }

    return quit-1;
}
