/* Kauiten kest RSAPublicEncrypt */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "crc.h"
#include "files.h"
#include "openkey.h"
#include "pubkey.h"

static NN_DIGIT NN_AddDigitMult PROTO_LIST 
  ((NN_DIGIT *, NN_DIGIT *, NN_DIGIT, NN_DIGIT *, unsigned int));
static NN_DIGIT NN_SubDigitMult PROTO_LIST 
  ((NN_DIGIT *, NN_DIGIT *, NN_DIGIT, NN_DIGIT *, unsigned int));

static unsigned int NN_DigitBits PROTO_LIST ((NN_DIGIT));

/* Decodes character string b into a, where character string is ordered
   from most to least significant.

   Lengths: a[digits], b[len].
   Assumes b[i] = 0 for i < len - digits * NN_DIGIT_LEN. (Otherwise most
   significant bytes are truncated.)
 */
static void NN_Decode (
                NN_DIGIT *a,
                unsigned int digits, 
unsigned char *b,
unsigned int len)
{
  NN_DIGIT t;
  int j;
  unsigned int i, u;
  
  for (i = 0, j = len - 1; i < digits && j >= 0; i++) {
    t = 0;
    for (u = 0; j >= 0 && u < NN_DIGIT_BITS; j--, u += 8)
      t |= ((NN_DIGIT)b[j]) << u;
    a[i] = t;
  }
  
  for (; i < digits; i++)
    a[i] = 0;
}

/* Encodes b into character string a, where character string is ordered
   from most to least significant.

   Lengths: a[len], b[digits].
   Assumes NN_Bits (b, digits) <= 8 * len. (Otherwise most significant
   digits are truncated.)
 */
static void NN_Encode (
unsigned char *a,
unsigned int digits,
NN_DIGIT *b,
unsigned int len)
{
  NN_DIGIT t;
  int j;
  unsigned int i, u;

  for (i = 0, j = len - 1; i < digits && j >= 0; i++) {
    t = b[i];
    for (u = 0; j >= 0 && u < NN_DIGIT_BITS; j--, u += 8)
      a[j] = (unsigned char)(t >> u);
  }

  for (; j >= 0; j--)
    a[j] = 0;
}

/* Assigns a = b.

   Lengths: a[digits], b[digits].
 */
static void NN_Assign (
NN_DIGIT *a, NN_DIGIT *b,
unsigned int digits)
{
  unsigned int i;

  for (i = 0; i < digits; i++)
    a[i] = b[i];
}

/* Assigns a = 0.

   Lengths: a[digits].
 */
static void NN_AssignZero (
NN_DIGIT *a,
unsigned int digits)
{
  unsigned int i;

  for (i = 0; i < digits; i++)
    a[i] = 0;
}

/* Computes a = b - c. Returns borrow.

   Lengths: a[digits], b[digits], c[digits].
 */
static NN_DIGIT NN_Sub (
NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT *c,
unsigned int digits)
{
  NN_DIGIT ai, borrow;
  unsigned int i;

  borrow = 0;

  for (i = 0; i < digits; i++) {
    if ((ai = b[i] - borrow) > (MAX_NN_DIGIT - borrow))
      ai = MAX_NN_DIGIT - c[i];
    else if ((ai -= c[i]) > (MAX_NN_DIGIT - c[i]))
      borrow = 1;
    else
      borrow = 0;
    a[i] = ai;
  }

  return (borrow);
}

/* Computes a = b * c.

   Lengths: a[2*digits], b[digits], c[digits].
   Assumes digits < MAX_NN_DIGITS.
 */
static void NN_Mult (
NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT *c,
unsigned int digits)
{
  NN_DIGIT t[2*MAX_NN_DIGITS];
  unsigned int bDigits, cDigits, i;

  NN_AssignZero (t, 2 * digits);
  
  bDigits = NN_Digits (b, digits);
  cDigits = NN_Digits (c, digits);

  for (i = 0; i < bDigits; i++)
    t[i+cDigits] += NN_AddDigitMult (&t[i], &t[i], b[i], c, cDigits);
  
  NN_Assign (a, t, 2 * digits);
  
  /* Zeroize potentially sensitive information.
   */
  /*memset ((POINTER)t, 0, sizeof (t));*/
}

/* Computes a = b * 2^c (i.e., shifts left c bits), returning carry.

   Lengths: a[digits], b[digits].
   Requires c < NN_DIGIT_BITS.
 */
static NN_DIGIT NN_LShift (
NN_DIGIT *a, NN_DIGIT *b,
unsigned int c, unsigned int digits)
{
  NN_DIGIT bi, carry;
  unsigned int i, t;
  
  if (c >= NN_DIGIT_BITS)
    return (0);
  
  t = NN_DIGIT_BITS - c;

  carry = 0;

  for (i = 0; i < digits; i++) {
    bi = b[i];
    a[i] = (bi << c) | carry;
    carry = c ? (bi >> t) : 0;
  }
  
  return (carry);
}

/* Computes a = c div 2^c (i.e., shifts right c bits), returning carry.

   Lengths: a[digits], b[digits].
   Requires: c < NN_DIGIT_BITS.
 */
static NN_DIGIT NN_RShift (
NN_DIGIT *a, NN_DIGIT *b,
unsigned int c, unsigned int digits)
{
  NN_DIGIT bi, carry;
  int i;
  unsigned int t;
  
  if (c >= NN_DIGIT_BITS)
    return (0);
  
  t = NN_DIGIT_BITS - c;

  carry = 0;

  for (i = digits - 1; i >= 0; i--) {
    bi = b[i];
    a[i] = (bi >> c) | carry;
    carry = c ? (bi << t) : 0;
  }
  
  return (carry);
}

/* Sets a = b / c, where a and c are digits.

   Lengths: b[2].
   Assumes b[1] < c and HIGH_HALF (c) > 0. For efficiency, c should be
   normalized.
 */
static void NN_DigitDiv (
NN_DIGIT *a, NN_DIGIT b[2], NN_DIGIT c)
{
  NN_DIGIT t[2], u, v;
  NN_HALF_DIGIT aHigh, aLow, cHigh, cLow;

  cHigh = (NN_HALF_DIGIT)HIGH_HALF (c);
  cLow = (NN_HALF_DIGIT)LOW_HALF (c);

  t[0] = b[0];
  t[1] = b[1];

  /* Underestimate high half of quotient and subtract.
   */
  if (cHigh == MAX_NN_HALF_DIGIT)
    aHigh = (NN_HALF_DIGIT)HIGH_HALF (t[1]);
  else
    aHigh = (NN_HALF_DIGIT)(t[1] / (cHigh + 1));
  u = (NN_DIGIT)aHigh * (NN_DIGIT)cLow;
  v = (NN_DIGIT)aHigh * (NN_DIGIT)cHigh;
  if ((t[0] -= TO_HIGH_HALF (u)) > (MAX_NN_DIGIT - TO_HIGH_HALF (u)))
    t[1]--;
  t[1] -= HIGH_HALF (u);
  t[1] -= v;

  /* Correct estimate.
   */
  while ((t[1] > cHigh) ||
         ((t[1] == cHigh) && (t[0] >= TO_HIGH_HALF (cLow)))) {
    if ((t[0] -= TO_HIGH_HALF (cLow)) > MAX_NN_DIGIT - TO_HIGH_HALF (cLow))
      t[1]--;
    t[1] -= cHigh;
    aHigh++;
  }

  /* Underestimate low half of quotient and subtract.
   */
  if (cHigh == MAX_NN_HALF_DIGIT)
    aLow = (NN_HALF_DIGIT)LOW_HALF (t[1]);
  else
    aLow = 
      (NN_HALF_DIGIT)((TO_HIGH_HALF (t[1]) + HIGH_HALF (t[0])) / (cHigh + 1));
  u = (NN_DIGIT)aLow * (NN_DIGIT)cLow;
  v = (NN_DIGIT)aLow * (NN_DIGIT)cHigh;
  if ((t[0] -= u) > (MAX_NN_DIGIT - u))
    t[1]--;
  if ((t[0] -= TO_HIGH_HALF (v)) > (MAX_NN_DIGIT - TO_HIGH_HALF (v)))
    t[1]--;
  t[1] -= HIGH_HALF (v);

  /* Correct estimate.
   */
  while ((t[1] > 0) || ((t[1] == 0) && t[0] >= c)) {
    if ((t[0] -= c) > (MAX_NN_DIGIT - c))
      t[1]--;
    aLow++;
  }
  
  *a = TO_HIGH_HALF (aHigh) + aLow;
}

/* Computes a = c div d and b = c mod d.

   Lengths: a[cDigits], b[dDigits], c[cDigits], d[dDigits].
   Assumes d > 0, cDigits < 2 * MAX_NN_DIGITS,
           dDigits < MAX_NN_DIGITS.
 */
static void NN_Div (
             NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT *c,
             unsigned int cDigits, 
             NN_DIGIT *d,
unsigned int dDigits)
{
  NN_DIGIT ai, cc[2*MAX_NN_DIGITS+1], dd[MAX_NN_DIGITS], t;
  int i;
  unsigned int ddDigits, shift;
  
  ddDigits = NN_Digits (d, dDigits);
  if (ddDigits == 0)
    return;
  
  /* Normalize operands.
   */
  shift = NN_DIGIT_BITS - NN_DigitBits (d[ddDigits-1]);
  NN_AssignZero (cc, ddDigits);
  cc[cDigits] = NN_LShift (cc, c, shift, cDigits);
  NN_LShift (dd, d, shift, ddDigits);
  t = dd[ddDigits-1];
  
  NN_AssignZero (a, cDigits);

  for (i = cDigits-ddDigits; i >= 0; i--) {
    /* Underestimate quotient digit and subtract.
     */
    if (t == MAX_NN_DIGIT)
      ai = cc[i+ddDigits];
    else
      NN_DigitDiv (&ai, &cc[i+ddDigits-1], t + 1);
    cc[i+ddDigits] -= NN_SubDigitMult (&cc[i], &cc[i], ai, dd, ddDigits);

    /* Correct estimate.
     */
    while (cc[i+ddDigits] || (NN_Cmp (&cc[i], dd, ddDigits) >= 0)) {
      ai++;
      cc[i+ddDigits] -= NN_Sub (&cc[i], &cc[i], dd, ddDigits);
    }
    
    a[i] = ai;
  }
  
  /* Restore result.
   */
  NN_AssignZero (b, dDigits);
  NN_RShift (b, cc, shift, ddDigits);

  /* Zeroize potentially sensitive information.
   */
  /*memset ((POINTER)cc, 0, sizeof (cc));
  memset ((POINTER)dd, 0, sizeof (dd));*/
}

/* Computes a = b mod c.

   Lengths: a[cDigits], b[bDigits], c[cDigits].
   Assumes c > 0, bDigits < 2 * MAX_NN_DIGITS, cDigits < MAX_NN_DIGITS.
 */
static void NN_Mod (
             NN_DIGIT *a, NN_DIGIT *b,
             unsigned int bDigits, 
             NN_DIGIT *c,
             unsigned int cDigits)
{
  NN_DIGIT t[2 * MAX_NN_DIGITS];
  
  NN_Div (t, a, b, bDigits, c, cDigits);
  
  /* Zeroize potentially sensitive information.
   */
  /*memset ((POINTER)t, 0, sizeof (t));*/
}

/* Computes a = b * c mod d.

   Lengths: a[digits], b[digits], c[digits], d[digits].
   Assumes d > 0, digits < MAX_NN_DIGITS.
 */
static void NN_ModMult (
NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT *c, NN_DIGIT *d,
unsigned int digits)
{
  NN_DIGIT t[2*MAX_NN_DIGITS];

  NN_Mult (t, b, c, digits);
  NN_Mod (a, t, 2 * digits, d, digits);
  
  /* Zeroize potentially sensitive information.
   */
  /*memset ((POINTER)t, 0, sizeof (t));*/
}

/* Computes a = b^c mod d.

   Lengths: a[dDigits], b[dDigits], c[cDigits], d[dDigits].
   Assumes d > 0, cDigits > 0, dDigits < MAX_NN_DIGITS.
 */
static void NN_ModExp (
                NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT *c,
                unsigned int cDigits, 
                NN_DIGIT *d,
unsigned int dDigits)
{
  NN_DIGIT bPower[3][MAX_NN_DIGITS], ci, t[MAX_NN_DIGITS];
  int i;
  unsigned int ciBits, j, s;

  /* Store b, b^2 mod d, and b^3 mod d.
   */
  NN_Assign (bPower[0], b, dDigits);
  NN_ModMult (bPower[1], bPower[0], b, d, dDigits);
  NN_ModMult (bPower[2], bPower[1], b, d, dDigits);
  
  NN_ASSIGN_DIGIT (t, 1, dDigits);

  cDigits = NN_Digits (c, cDigits);
  for (i = cDigits - 1; i >= 0; i--) {
    ci = c[i];
    ciBits = NN_DIGIT_BITS;
    
    /* Scan past leading zero bits of most significant digit.
     */
    if (i == (int)(cDigits - 1)) {
      while (! DIGIT_2MSB (ci)) {
        ci <<= 2;
        ciBits -= 2;
      }
    }

    for (j = 0; j < ciBits; j += 2, ci <<= 2) {
      /* Compute t = t^4 * b^s mod d, where s = two MSB's of ci.
       */
      NN_ModMult (t, t, t, d, dDigits);
      NN_ModMult (t, t, t, d, dDigits);
      if ((s = DIGIT_2MSB (ci)) != 0)
        NN_ModMult (t, t, bPower[s-1], d, dDigits);
    }
  }
  
  NN_Assign (a, t, dDigits);
  
  /* Zeroize potentially sensitive information.
   */
  /*memset ((POINTER)bPower, 0, sizeof (bPower));
  memset ((POINTER)t, 0, sizeof (t));*/
}

/* Returns sign of a - b.

   Lengths: a[digits], b[digits].
 */
static int NN_Cmp (
NN_DIGIT *a, NN_DIGIT *b,
unsigned int digits)
{
  int i;
  
  for (i = digits - 1; i >= 0; i--) {
    if (a[i] > b[i])
      return (1);
    if (a[i] < b[i])
      return (-1);
  }

  return (0);
}

/* Returns the significant length of a in digits.

   Lengths: a[digits].
 */
static unsigned int NN_Digits (NN_DIGIT *a, unsigned int digits)
{
  int i;
  
  for (i = digits - 1; i >= 0; i--)
    if (a[i])
      break;

  return (i + 1);
}

/* Computes a = b * c, where b and c are digits.

   Lengths: a[2].
   */
#ifdef __aDOS__

void NN_DigitMult (NN_DIGIT a[2], NN_DIGIT b, NN_DIGIT c);
#pragma aux NN_DigitMult = \
    "" /* a[0] = (b & 0xffff) * (c & 0xffff) */ \
    "push bp" \
    "mov si, ax" /* si = low(b) */ \
    "mov bp, dx" /* bp = low(c) */ \
    "mul dx" /* low(b) * low(c) */ \
    "mov es:[di], ax" /* a[0] = result */ \
    "mov es:[di+2], dx" \
    "" /* t = (b & 0xffff) * (c >> 16) */ \
    "mov ax, si" \
    "mul cx" /* low(b) * hi(c) */ \
    "mov si, ax" /* bp:si = result */ \
    "mov ax, bp" \
    "mov bp, dx" \
    "" /* u = (b >> 16) * (c & 0xffff) */ \
    "mul bx" /* low(c) * hi(b) */ \
    "push ax" /* result to stack - korvaa xchg:ll.. */ \
    "push dx" \
    "" /* a[1] = (b >> 16) * (c >> 16) */ \
    "mov ax, bx" \
    "mul cx" /* hi(b) * hi(c) */ \
    "mov es:[di+4], ax" /* a[1] = result */ \
    "mov es:[di+6], dx" \
    "pop dx" /* ax:dx = result of low(c) * hi(b) */ \
    "pop ax" \
    "" /* t += u    -    t = bp:si, u = dx:ax */ \
    "add si, ax" \
    "adc bp, 0" \
    "add bp, dx" \
    "" /* if (t < u) a[1] += 65536 */ \
    "cmp bp, dx" \
    "ja __over1" \
    "cmp bp, dx" \
    "jb __ok1" \
    "cmp si, ax" \
    "jnb __over1" \
"__ok1:" \
    "inc word ptr es:[di+6]" \
"__over1:" \
    "" /* u = t << 16 - tm pois ja si:t voi kytt dx:n tilalla.. */ \
    "mov dx, si" \
    "" /* a[1] += (t >> 16) */ \
    "add es:[di+4], bp" \
    "adc es:[di+6], 0" \
    "" /* a[0] += u (low part of u is 0) */ \
    "add es:[di+2], dx" \
    "" /* if (a[0] < u) a[1]++; */ \
    "cmp es:[di+2], dx" \
    "jnb __over2" \
    "inc word ptr es:[di+4]" \
    "adc es:[di+6], 0" \
"__over2:" \
    "pop bp" \
    modify [si] \
    parm [es di] [ax bx] [dx cx];

#else
void NN_DigitMult (NN_DIGIT a[2], NN_DIGIT b, NN_DIGIT c)
{
    unsigned long t, u;

    a[0] = (b & 0xffff) * (c & 0xffff);
    t = (b & 0xffff) * (c >> 16);
    u = (b >> 16) * (c & 0xffff);
    a[1] = (b >> 16) * (c >> 16);

    t += u;
    if (t < u) a[1] += 65536;
    u = t << 16;
    a[1] += (t >> 16);

    a[0] += u;
    if (a[0] < u) a[1]++;
}
#endif

/* Computes a = b + c*d, where c is a digit. Returns carry.

   Lengths: a[digits], b[digits], d[digits].
 */
static NN_DIGIT NN_AddDigitMult (
NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT c, NN_DIGIT *d,
unsigned int digits)
{
  NN_DIGIT carry, t[2];
  unsigned int i;

  if (c == 0)
    return (0);

  carry = 0;
  for (i = 0; i < digits; i++) {
    NN_DigitMult (t, c, d[i]);
    if ((a[i] = b[i] + carry) < carry)
      carry = 1;
    else
      carry = 0;
    if ((a[i] += t[0]) < t[0])
      carry++;
    carry += t[1];
  }
  
  return (carry);
}

/* Computes a = b - c*d, where c is a digit. Returns borrow.

   Lengths: a[digits], b[digits], d[digits].
 */
static NN_DIGIT NN_SubDigitMult (
NN_DIGIT *a, NN_DIGIT *b, NN_DIGIT c, NN_DIGIT *d,
unsigned int digits)
{
  NN_DIGIT borrow, t[2];
  unsigned int i;

  if (c == 0)
    return (0);

  borrow = 0;
  for (i = 0; i < digits; i++) {
    NN_DigitMult (t, c, d[i]);
    if ((a[i] = b[i] - borrow) > (MAX_NN_DIGIT - borrow))
      borrow = 1;
    else
      borrow = 0;
    if ((a[i] -= t[0]) > (MAX_NN_DIGIT - t[0]))
      borrow++;
    borrow += t[1];
  }
  
  return (borrow);
}

/* Returns the significant length of a in bits, where a is a digit.
 */
static unsigned int NN_DigitBits (NN_DIGIT a)
{
  unsigned int i;
  
  for (i = 0; i < NN_DIGIT_BITS; i++, a >>= 1)
    if (a == 0)
      break;
    
  return (i);
}

static UINT2 BYTE_BIT[8] = {
  0200, 0100, 040, 020, 010, 04, 02, 01
};

static UINT4 BIG_BYTE[24] = {
  0x800000L, 0x400000L, 0x200000L, 0x100000L,
  0x80000L,  0x40000L,  0x20000L,  0x10000L,
  0x8000L,   0x4000L,   0x2000L,   0x1000L,
  0x800L,    0x400L,    0x200L,    0x100L,
  0x80L,     0x40L,     0x20L,     0x10L,
  0x8L,      0x4L,      0x2L,      0x1L
};

static unsigned char PC1[56] = {
  56, 48, 40, 32, 24, 16,  8,      0, 57, 49, 41, 33, 25, 17,
   9,  1, 58, 50, 42, 34, 26,     18, 10,  2, 59, 51, 43, 35,
  62, 54, 46, 38, 30, 22, 14,      6, 61, 53, 45, 37, 29, 21,
  13,  5, 60, 52, 44, 36, 28,     20, 12,  4, 27, 19, 11,  3
};

static unsigned char TOTAL_ROTATIONS[16] = {
  1, 2, 4, 6, 8, 10, 12, 14, 15, 17, 19, 21, 23, 25, 27, 28
};

static unsigned char PC2[48] = {
  13, 16, 10, 23,  0,  4,  2, 27, 14,  5, 20,  9,
  22, 18, 11,  3, 25,  7, 15,  6, 26, 19, 12,  1,
  40, 51, 30, 36, 46, 54, 29, 39, 50, 44, 32, 47,
  43, 48, 38, 55, 33, 52, 45, 41, 49, 35, 28, 31
};

static UINT4 SP1[64] = {
  0x01010400L, 0x00000000L, 0x00010000L, 0x01010404L,
  0x01010004L, 0x00010404L, 0x00000004L, 0x00010000L,
  0x00000400L, 0x01010400L, 0x01010404L, 0x00000400L,
  0x01000404L, 0x01010004L, 0x01000000L, 0x00000004L,
  0x00000404L, 0x01000400L, 0x01000400L, 0x00010400L,
  0x00010400L, 0x01010000L, 0x01010000L, 0x01000404L,
  0x00010004L, 0x01000004L, 0x01000004L, 0x00010004L,
  0x00000000L, 0x00000404L, 0x00010404L, 0x01000000L,
  0x00010000L, 0x01010404L, 0x00000004L, 0x01010000L,
  0x01010400L, 0x01000000L, 0x01000000L, 0x00000400L,
  0x01010004L, 0x00010000L, 0x00010400L, 0x01000004L,
  0x00000400L, 0x00000004L, 0x01000404L, 0x00010404L,
  0x01010404L, 0x00010004L, 0x01010000L, 0x01000404L,
  0x01000004L, 0x00000404L, 0x00010404L, 0x01010400L,
  0x00000404L, 0x01000400L, 0x01000400L, 0x00000000L,
  0x00010004L, 0x00010400L, 0x00000000L, 0x01010004L
};

static UINT4 SP2[64] = {
  0x80108020L, 0x80008000L, 0x00008000L, 0x00108020L,
  0x00100000L, 0x00000020L, 0x80100020L, 0x80008020L,
  0x80000020L, 0x80108020L, 0x80108000L, 0x80000000L,
  0x80008000L, 0x00100000L, 0x00000020L, 0x80100020L,
  0x00108000L, 0x00100020L, 0x80008020L, 0x00000000L,
  0x80000000L, 0x00008000L, 0x00108020L, 0x80100000L,
  0x00100020L, 0x80000020L, 0x00000000L, 0x00108000L,
  0x00008020L, 0x80108000L, 0x80100000L, 0x00008020L,
  0x00000000L, 0x00108020L, 0x80100020L, 0x00100000L,
  0x80008020L, 0x80100000L, 0x80108000L, 0x00008000L,
  0x80100000L, 0x80008000L, 0x00000020L, 0x80108020L,
  0x00108020L, 0x00000020L, 0x00008000L, 0x80000000L,
  0x00008020L, 0x80108000L, 0x00100000L, 0x80000020L,
  0x00100020L, 0x80008020L, 0x80000020L, 0x00100020L,
  0x00108000L, 0x00000000L, 0x80008000L, 0x00008020L,
  0x80000000L, 0x80100020L, 0x80108020L, 0x00108000L
};

static UINT4 SP3[64] = {
  0x00000208L, 0x08020200L, 0x00000000L, 0x08020008L,
  0x08000200L, 0x00000000L, 0x00020208L, 0x08000200L,
  0x00020008L, 0x08000008L, 0x08000008L, 0x00020000L,
  0x08020208L, 0x00020008L, 0x08020000L, 0x00000208L,
  0x08000000L, 0x00000008L, 0x08020200L, 0x00000200L,
  0x00020200L, 0x08020000L, 0x08020008L, 0x00020208L,
  0x08000208L, 0x00020200L, 0x00020000L, 0x08000208L,
  0x00000008L, 0x08020208L, 0x00000200L, 0x08000000L,
  0x08020200L, 0x08000000L, 0x00020008L, 0x00000208L,
  0x00020000L, 0x08020200L, 0x08000200L, 0x00000000L,
  0x00000200L, 0x00020008L, 0x08020208L, 0x08000200L,
  0x08000008L, 0x00000200L, 0x00000000L, 0x08020008L,
  0x08000208L, 0x00020000L, 0x08000000L, 0x08020208L,
  0x00000008L, 0x00020208L, 0x00020200L, 0x08000008L,
  0x08020000L, 0x08000208L, 0x00000208L, 0x08020000L,
  0x00020208L, 0x00000008L, 0x08020008L, 0x00020200L
};

static UINT4 SP4[64] = {
  0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L,
  0x00802080L, 0x00800081L, 0x00800001L, 0x00002001L,
  0x00000000L, 0x00802000L, 0x00802000L, 0x00802081L,
  0x00000081L, 0x00000000L, 0x00800080L, 0x00800001L,
  0x00000001L, 0x00002000L, 0x00800000L, 0x00802001L,
  0x00000080L, 0x00800000L, 0x00002001L, 0x00002080L,
  0x00800081L, 0x00000001L, 0x00002080L, 0x00800080L,
  0x00002000L, 0x00802080L, 0x00802081L, 0x00000081L,
  0x00800080L, 0x00800001L, 0x00802000L, 0x00802081L,
  0x00000081L, 0x00000000L, 0x00000000L, 0x00802000L,
  0x00002080L, 0x00800080L, 0x00800081L, 0x00000001L,
  0x00802001L, 0x00002081L, 0x00002081L, 0x00000080L,
  0x00802081L, 0x00000081L, 0x00000001L, 0x00002000L,
  0x00800001L, 0x00002001L, 0x00802080L, 0x00800081L,
  0x00002001L, 0x00002080L, 0x00800000L, 0x00802001L,
  0x00000080L, 0x00800000L, 0x00002000L, 0x00802080L
};

static UINT4 SP5[64] = {
  0x00000100L, 0x02080100L, 0x02080000L, 0x42000100L,
  0x00080000L, 0x00000100L, 0x40000000L, 0x02080000L,
  0x40080100L, 0x00080000L, 0x02000100L, 0x40080100L,
  0x42000100L, 0x42080000L, 0x00080100L, 0x40000000L,
  0x02000000L, 0x40080000L, 0x40080000L, 0x00000000L,
  0x40000100L, 0x42080100L, 0x42080100L, 0x02000100L,
  0x42080000L, 0x40000100L, 0x00000000L, 0x42000000L,
  0x02080100L, 0x02000000L, 0x42000000L, 0x00080100L,
  0x00080000L, 0x42000100L, 0x00000100L, 0x02000000L,
  0x40000000L, 0x02080000L, 0x42000100L, 0x40080100L,
  0x02000100L, 0x40000000L, 0x42080000L, 0x02080100L,
  0x40080100L, 0x00000100L, 0x02000000L, 0x42080000L,
  0x42080100L, 0x00080100L, 0x42000000L, 0x42080100L,
  0x02080000L, 0x00000000L, 0x40080000L, 0x42000000L,
  0x00080100L, 0x02000100L, 0x40000100L, 0x00080000L,
  0x00000000L, 0x40080000L, 0x02080100L, 0x40000100L
};

static UINT4 SP6[64] = {
  0x20000010L, 0x20400000L, 0x00004000L, 0x20404010L,
  0x20400000L, 0x00000010L, 0x20404010L, 0x00400000L,
  0x20004000L, 0x00404010L, 0x00400000L, 0x20000010L,
  0x00400010L, 0x20004000L, 0x20000000L, 0x00004010L,
  0x00000000L, 0x00400010L, 0x20004010L, 0x00004000L,
  0x00404000L, 0x20004010L, 0x00000010L, 0x20400010L,
  0x20400010L, 0x00000000L, 0x00404010L, 0x20404000L,
  0x00004010L, 0x00404000L, 0x20404000L, 0x20000000L,
  0x20004000L, 0x00000010L, 0x20400010L, 0x00404000L,
  0x20404010L, 0x00400000L, 0x00004010L, 0x20000010L,
  0x00400000L, 0x20004000L, 0x20000000L, 0x00004010L,
  0x20000010L, 0x20404010L, 0x00404000L, 0x20400000L,
  0x00404010L, 0x20404000L, 0x00000000L, 0x20400010L,
  0x00000010L, 0x00004000L, 0x20400000L, 0x00404010L,
  0x00004000L, 0x00400010L, 0x20004010L, 0x00000000L,
  0x20404000L, 0x20000000L, 0x00400010L, 0x20004010L
};

static UINT4 SP7[64] = {
  0x00200000L, 0x04200002L, 0x04000802L, 0x00000000L,
  0x00000800L, 0x04000802L, 0x00200802L, 0x04200800L,
  0x04200802L, 0x00200000L, 0x00000000L, 0x04000002L,
  0x00000002L, 0x04000000L, 0x04200002L, 0x00000802L,
  0x04000800L, 0x00200802L, 0x00200002L, 0x04000800L,
  0x04000002L, 0x04200000L, 0x04200800L, 0x00200002L,
  0x04200000L, 0x00000800L, 0x00000802L, 0x04200802L,
  0x00200800L, 0x00000002L, 0x04000000L, 0x00200800L,
  0x04000000L, 0x00200800L, 0x00200000L, 0x04000802L,
  0x04000802L, 0x04200002L, 0x04200002L, 0x00000002L,
  0x00200002L, 0x04000000L, 0x04000800L, 0x00200000L,
  0x04200800L, 0x00000802L, 0x00200802L, 0x04200800L,
  0x00000802L, 0x04000002L, 0x04200802L, 0x04200000L,
  0x00200800L, 0x00000000L, 0x00000002L, 0x04200802L,
  0x00000000L, 0x00200802L, 0x04200000L, 0x00000800L,
  0x04000002L, 0x04000800L, 0x00000800L, 0x00200002L
};

static UINT4 SP8[64] = {
  0x10001040L, 0x00001000L, 0x00040000L, 0x10041040L,
  0x10000000L, 0x10001040L, 0x00000040L, 0x10000000L,
  0x00040040L, 0x10040000L, 0x10041040L, 0x00041000L,
  0x10041000L, 0x00041040L, 0x00001000L, 0x00000040L,
  0x10040000L, 0x10000040L, 0x10001000L, 0x00001040L,
  0x00041000L, 0x00040040L, 0x10040040L, 0x10041000L,
  0x00001040L, 0x00000000L, 0x00000000L, 0x10040040L,
  0x10000040L, 0x10001000L, 0x00041040L, 0x00040000L,
  0x00041040L, 0x00040000L, 0x10041000L, 0x00001000L,
  0x00000040L, 0x10040040L, 0x00001000L, 0x00041040L,
  0x10001000L, 0x00000040L, 0x10000040L, 0x10040000L,
  0x10040040L, 0x10000000L, 0x00040000L, 0x10001040L,
  0x00000000L, 0x10041040L, 0x00040040L, 0x10000040L,
  0x10040000L, 0x10001000L, 0x10001040L, 0x00000000L,
  0x10041040L, 0x00041000L, 0x00041000L, 0x00001040L,
  0x00001040L, 0x00040040L, 0x10000000L, 0x10041000L
};

#define RANDOM_BYTES_NEEDED 256

static int R_RandomInit (R_RANDOM_STRUCT *randomStruct)
{
  randomStruct->bytesNeeded = RANDOM_BYTES_NEEDED;
  memset ((POINTER)randomStruct->state, 0, sizeof (randomStruct->state));
  randomStruct->outputAvailable = 0;
  
  return (0);
}

static int R_RandomUpdate (
R_RANDOM_STRUCT *randomStruct,                          /* random structure */
unsigned char *block,                          /* block of values to mix in */
unsigned int blockLen)                                   /* length of block */
{
  MD5_CTX context;
  unsigned char digest[16];
  unsigned int i, x;
  
  MD5Init (&context);
  MD5Update (&context, block, blockLen);
  MD5Final (digest, &context);

  /* add digest to state */
  x = 0;
  for (i = 0; i < 16; i++) {
    x += randomStruct->state[15-i] + digest[15-i];
    randomStruct->state[15-i] = (unsigned char)x;
    x >>= 8;
  }
  
  if (randomStruct->bytesNeeded < blockLen)
    randomStruct->bytesNeeded = 0;
  else
    randomStruct->bytesNeeded -= blockLen;
  
  /* Zeroize sensitive information.
   */
  /*memset ((POINTER)digest, 0, sizeof (digest));
  x = 0;*/
  
  return (0);
}

static int R_GetRandomBytesNeeded (
unsigned int *bytesNeeded,                 /* number of mix-in bytes needed */
R_RANDOM_STRUCT *randomStruct)                          /* random structure */
{
  *bytesNeeded = randomStruct->bytesNeeded;
  
  return (0);
}

static int R_GenerateBytes (
unsigned char *block,                                             /* block */
unsigned int blockLen,                                   /* length of block */
R_RANDOM_STRUCT *randomStruct)                          /* random structure */
{
  MD5_CTX context;
  unsigned int available, i;
  
  if (randomStruct->bytesNeeded)
    return (RE_NEED_RANDOM);
  
  available = randomStruct->outputAvailable;
  
  while (blockLen > available) {
    memcpy
      ((POINTER)block, (POINTER)&randomStruct->output[16-available],
       available);
    block += available;
    blockLen -= available;

    /* generate new output */
    MD5Init (&context);
    MD5Update (&context, randomStruct->state, 16);
    MD5Final (randomStruct->output, &context);
    available = 16;

    /* increment state */
    for (i = 0; i < 16; i++)
      if (randomStruct->state[15-i]++)
        break;
  }

  memcpy ((POINTER)block, (POINTER)&randomStruct->output[16-available], blockLen);
  randomStruct->outputAvailable = available - blockLen;

  return (0);
}

static void InitRandomStruct (R_RANDOM_STRUCT *randomStruct)
{
    static unsigned char seedByte = 0;
    unsigned int bytesNeeded;

    R_RandomInit (randomStruct);

    /* Initialize with all zero seed bytes, which will not yield an actual
       random number output.
    */
    for (;;)
    {
        R_GetRandomBytesNeeded (&bytesNeeded, randomStruct);
        if (bytesNeeded == 0)
            break;

        R_RandomUpdate (randomStruct, &seedByte, 1);
    }
}

//#if defined (__DOS__) && !defined (__386__)
#if 0
void Pack (UINT4 *into, unsigned char *outof);
#pragma aux Pack = \
    "push ds" \
    "mov ds, ax" \
    "mov ch, [si]" \
    "mov cl, [si+1]" \
    "mov ah, [si+2]" \
    "mov al, [si+3]" \
    "mov es:[bx], ax" \
    "mov es:[bx+2], cx" \
    "mov ch, [si+4]" \
    "mov cl, [si+5]" \
    "mov ah, [si+6]" \
    "mov al, [si+7]" \
    "mov es:[bx+4], ax" \
    "mov es:[bx+6], cx" \
    "pop ds" \
    modify [cx] \
    parm [es bx] [ax si];

void Unpack (unsigned char *into, UINT4 *outof);
#pragma aux Unpack = \
    "push ds" \
    "mov ds, ax" \
    "mov ch, [di]" \
    "mov cl, [di+1]" \
    "mov ah, [di+2]" \
    "mov al, [di+3]" \
    "mov es:[bx], ax" \
    "mov es:[bx+2], cx" \
    "mov ch, [di+4]" \
    "mov cl, [di+5]" \
    "mov ah, [di+6]" \
    "mov al, [di+7]" \
    "mov es:[bx+4], ax" \
    "mov es:[bx+6], cx" \
    "pop ds" \
    modify [cx] \
    parm [es bx] [ax di];

#else

static void Pack (UINT4 *into, unsigned char *outof)
{
  *into    = (*outof++ & 0xffL) << 24;
  *into   |= (*outof++ & 0xffL) << 16;
  *into   |= (*outof++ & 0xffL) << 8;
  *into++ |= (*outof++ & 0xffL);
  *into    = (*outof++ & 0xffL) << 24;
  *into   |= (*outof++ & 0xffL) << 16;
  *into   |= (*outof++ & 0xffL) << 8;
  *into   |= (*outof   & 0xffL);
}

static void Unpack (unsigned char *into, UINT4 *outof)
{
  *into++ = (unsigned char)((*outof >> 24) & 0xffL);
  *into++ = (unsigned char)((*outof >> 16) & 0xffL);
  *into++ = (unsigned char)((*outof >>  8) & 0xffL);
  *into++ = (unsigned char)( *outof++      & 0xffL);
  *into++ = (unsigned char)((*outof >> 24) & 0xffL);
  *into++ = (unsigned char)((*outof >> 16) & 0xffL);
  *into++ = (unsigned char)((*outof >>  8) & 0xffL);
  *into   = (unsigned char)( *outof        & 0xffL);
}
#endif

static void CookKey (
UINT4 *subkeys,
UINT4 *kn,
int encrypt)
{
  UINT4 *cooked, *raw0, *raw1;
  int increment;
  unsigned int i;

  raw1 = kn;
  cooked = encrypt ? subkeys : &subkeys[30];
  increment = encrypt ? 1 : -3;

  for (i = 0; i < 16; i++, raw1++) {
    raw0 = raw1++;
    *cooked    = (*raw0 & 0x00fc0000L) << 6;
    *cooked   |= (*raw0 & 0x00000fc0L) << 10;
    *cooked   |= (*raw1 & 0x00fc0000L) >> 10;
    *cooked++ |= (*raw1 & 0x00000fc0L) >> 6;
    *cooked    = (*raw0 & 0x0003f000L) << 12;
    *cooked   |= (*raw0 & 0x0000003fL) << 16;
    *cooked   |= (*raw1 & 0x0003f000L) >> 4;
    *cooked   |= (*raw1 & 0x0000003fL);
    cooked += increment;
  }
}

static void DESKey (
UINT4 subkeys[32],
unsigned char key[8],
int encrypt)
{
  UINT4 kn[32];
  int i, j, l, m, n;
  unsigned char pc1m[56], pcr[56];

  for (j = 0; j < 56; j++) {
    l = PC1[j];
    m = l & 07;
    pc1m[j] = (unsigned char)((key[l >> 3] & BYTE_BIT[m]) ? 1 : 0);
  }
  for (i = 0; i < 16; i++) {
    m = i << 1;
    n = m + 1;
    kn[m] = kn[n] = 0L;
    for (j = 0; j < 28; j++) {
      l = j + TOTAL_ROTATIONS[i];
      if (l < 28)
        pcr[j] = pc1m[l];
      else
        pcr[j] = pc1m[l - 28];
    }
    for (j = 28; j < 56; j++) {
      l = j + TOTAL_ROTATIONS[i];
      if (l < 56)
        pcr[j] = pc1m[l];
      else
        pcr[j] = pc1m[l - 28];
    }
    for (j = 0; j < 24; j++) {
      if (pcr[PC2[j]])
        kn[m] |= BIG_BYTE[j];
      if (pcr[PC2[j+24]])
        kn[n] |= BIG_BYTE[j];
    }
  }
  CookKey (subkeys, kn, encrypt);

  /* Zeroize sensitive information.
   */
  /*memset ((POINTER)pc1m, 0, sizeof (pc1m));
  memset ((POINTER)pcr, 0, sizeof (pcr));
  memset ((POINTER)kn, 0, sizeof (kn));*/
}

static void DESFunction (
UINT4 *block,
UINT4 *subkeys)
{
  register UINT4 fval, work, right, left;
  register int round;
  
  left = block[0];
  right = block[1];
  work = ((left >> 4) ^ right) & 0x0f0f0f0fL;
  right ^= work;
  left ^= (work << 4);
  work = ((left >> 16) ^ right) & 0x0000ffffL;
  right ^= work;
  left ^= (work << 16);
  work = ((right >> 2) ^ left) & 0x33333333L;
  left ^= work;
  right ^= (work << 2);
  work = ((right >> 8) ^ left) & 0x00ff00ffL;
  left ^= work;
  right ^= (work << 8);
  right = ((right << 1) | ((right >> 31) & 1L)) & 0xffffffffL;
  work = (left ^ right) & 0xaaaaaaaaL;
  left ^= work;
  right ^= work;
  left = ((left << 1) | ((left >> 31) & 1L)) & 0xffffffffL;
  
  for (round = 0; round < 8; round++) {
    work  = (right << 28) | (right >> 4);
    work ^= *subkeys++;
    fval  = SP7[ work        & 0x3fL];
    fval |= SP5[(work >>  8) & 0x3fL];
    fval |= SP3[(work >> 16) & 0x3fL];
    fval |= SP1[(work >> 24) & 0x3fL];
    work  = right ^ *subkeys++;
    fval |= SP8[ work        & 0x3fL];
    fval |= SP6[(work >>  8) & 0x3fL];
    fval |= SP4[(work >> 16) & 0x3fL];
    fval |= SP2[(work >> 24) & 0x3fL];
    left ^= fval;
    work  = (left << 28) | (left >> 4);
    work ^= *subkeys++;
    fval  = SP7[ work        & 0x3fL];
    fval |= SP5[(work >>  8) & 0x3fL];
    fval |= SP3[(work >> 16) & 0x3fL];
    fval |= SP1[(work >> 24) & 0x3fL];
    work  = left ^ *subkeys++;
    fval |= SP8[ work        & 0x3fL];
    fval |= SP6[(work >>  8) & 0x3fL];
    fval |= SP4[(work >> 16) & 0x3fL];
    fval |= SP2[(work >> 24) & 0x3fL];
    right ^= fval;
  }
  
  right = (right << 31) | (right >> 1);
  work = (left ^ right) & 0xaaaaaaaaL;
  left ^= work;
  right ^= work;
  left = (left << 31) | (left >> 1);
  work = ((left >> 8) ^ right) & 0x00ff00ffL;
  right ^= work;
  left ^= (work << 8);
  work = ((left >> 2) ^ right) & 0x33333333L;
  right ^= work;
  left ^= (work << 2);
  work = ((right >> 16) ^ left) & 0x0000ffffL;
  left ^= work;
  right ^= (work << 16);
  work = ((right >> 4) ^ left) & 0x0f0f0f0fL;
  left ^= work;
  right ^= (work << 4);
  *block++ = right;
  *block = left;
}
/* Initialize context.  Caller must zeroize the context when finished.
 */
static void DES_CBCInit (
DES_CBC_CTX *context,                                            /* context */
unsigned char key[8],                                                /* key */
unsigned char iv[8],                                 /* initializing vector */
int encrypt)                     /* encrypt flag (1 = encrypt, 0 = decrypt) */
{  
  /* Copy encrypt flag to context.
   */
  context->encrypt = encrypt;

  /* Pack initializing vector into context.
   */
  Pack (context->iv, iv);

  /* Save the IV for use in Restart */
  context->originalIV[0] = context->iv[0];
  context->originalIV[1] = context->iv[1];

  /* Precompute key schedule
   */
  DESKey (context->subkeys, key, encrypt);
}

/* DES-CBC block update operation. Continues a DES-CBC encryption
   operation, processing eight-byte message blocks, and updating
   the context.
 */
static int DES_CBCUpdate (
DES_CBC_CTX *context,                                            /* context */
unsigned char *output,                                      /* output block */
unsigned char *input,                                        /* input block */
unsigned int len)                      /* length of input and output blocks */
{
  UINT4 inputBlock[2], work[2];
  unsigned int i;
  
  if (len % 8)
    return (RE_LEN);

  for (i = 0; i < len/8; i++) {
    Pack (inputBlock, &input[8*i]);
        
    /* Chain if encrypting.
     */
    if (context->encrypt) {
      work[0] = inputBlock[0] ^ context->iv[0];
      work[1] = inputBlock[1] ^ context->iv[1];
    }
    else {
      work[0] = inputBlock[0];
      work[1] = inputBlock[1];         
    }

    DESFunction (work, context->subkeys);

    /* Chain if decrypting, then update IV.
     */
    if (context->encrypt) {
      context->iv[0] = work[0];
      context->iv[1] = work[1];
    }
    else {
      work[0] ^= context->iv[0];
      work[1] ^= context->iv[1];
      context->iv[0] = inputBlock[0];
      context->iv[1] = inputBlock[1];
    }
    Unpack (&output[8*i], work);
  }
  
  /* Zeroize sensitive information.
   */
  /*memset ((POINTER)inputBlock, 0, sizeof (inputBlock));
  memset ((POINTER)work, 0, sizeof (work));*/
  
  return (0);
}

/* Raw RSA public-key operation. Output has same length as modulus.

   Assumes inputLen < length of modulus.
   Requires input < modulus.
 */
static int RSAPublicBlock (
unsigned char *output,                                      /* output block */
unsigned int *outputLen,                          /* length of output block */
unsigned char *input,                                        /* input block */
unsigned int inputLen,                             /* length of input block */
R_RSA_PUBLIC_KEY *publicKey)                              /* RSA public key */
{
  NN_DIGIT c[MAX_NN_DIGITS], e[MAX_NN_DIGITS], m[MAX_NN_DIGITS],
    n[MAX_NN_DIGITS];
  unsigned int eDigits, nDigits;

  NN_Decode (m, MAX_NN_DIGITS, input, inputLen);
  NN_Decode (n, MAX_NN_DIGITS, publicKey->modulus, MAX_RSA_MODULUS_LEN);
  NN_Decode (e, MAX_NN_DIGITS, publicKey->exponent, MAX_RSA_MODULUS_LEN);
  nDigits = NN_Digits (n, MAX_NN_DIGITS);
  eDigits = NN_Digits (e, MAX_NN_DIGITS);
  
  if (NN_Cmp (m, n, nDigits) >= 0)
    return (RE_DATA);
  
  /* Compute c = m^e mod n.
   */
  NN_ModExp (c, m, e, eDigits, n, nDigits);

  *outputLen = (publicKey->bits + 7) / 8;
  NN_Encode (output, *outputLen, c, nDigits);
  
  /* Zeroize sensitive information.
   */
  /*memset ((POINTER)c, 0, sizeof (c));
  memset ((POINTER)m, 0, sizeof (m));*/

  return (0);
}

/* RSA public-key encryption, according to PKCS #1.
 */
static int RSAPublicEncrypt(
unsigned char *output,                                      /* output block */
unsigned int *outputLen,                          /* length of output block */
unsigned char *input,                                        /* input block */
unsigned int inputLen,                             /* length of input block */
R_RSA_PUBLIC_KEY *publicKey,                              /* RSA public key */
R_RANDOM_STRUCT *randomStruct)                          /* random structure */
{
  int status;
  unsigned char byte, pkcsBlock[MAX_RSA_MODULUS_LEN];
  unsigned int i, modulusLen;
  
  modulusLen = (publicKey->bits + 7) / 8;
  if (inputLen + 11 > modulusLen)
    return (RE_LEN);
  
  pkcsBlock[0] = 0;
  /* block type 2 */
  pkcsBlock[1] = 2;

  for (i = 2; i < modulusLen - inputLen - 1; i++) {
    /* Find nonzero random byte.
     */
    do {
      R_GenerateBytes (&byte, 1, randomStruct);
    } while (byte == 0);
    pkcsBlock[i] = byte;
  }
  /* separator */
  pkcsBlock[i++] = 0;
  
  memcpy ((POINTER)&pkcsBlock[i], (POINTER)input, inputLen);
  
  status = RSAPublicBlock (output, outputLen, pkcsBlock, modulusLen, publicKey);
  
  /* Zeroize sensitive information.
   */
  /*byte = 0;
  memset ((POINTER)pkcsBlock, 0, sizeof (pkcsBlock));*/
  
  return (status);
}

static int SealInit (
R_ENVELOPE_CTX *context,                                     /* new context */
unsigned char *encryptedKey,                              /* encrypted keys */
unsigned int *encryptedKeyLen,                   /* length of encrypted key */
R_RSA_PUBLIC_KEY *publicKey,                                 /* public keys */
R_RANDOM_STRUCT *randomStruct)                          /* random structure */
{
    unsigned char key[8], iv[8];

    if (R_GenerateBytes (key, 8, randomStruct) != 0) return 0;
    if (R_GenerateBytes (iv, 8, randomStruct) != 0) return 0;

    DES_CBCInit (&context->cipherContext.des, key, iv, 1);

    if (RSAPublicEncrypt (encryptedKey, encryptedKeyLen, key, 8,
                          publicKey, randomStruct))
        return 0;

    context->bufferLen = 0;

    return 1;
}

/* Assume partOut buffer is at least partInLen + 7, since this may flush
     buffered input.
 */
static void SealUpdate (
R_ENVELOPE_CTX *context,                                         /* context */
unsigned char *partOut,                         /* next encrypted data part */
unsigned int *partOutLen,             /* length of next encrypted data part */
unsigned char *partIn,                                    /* next data part */
unsigned int partInLen)                         /* length of next data part */
{
    unsigned int tempLen;

    tempLen = 8 - context->bufferLen;
    if (partInLen < tempLen)
    {
        /* Just accumulate into buffer. */
        memcpy ((POINTER)(context->buffer + context->bufferLen), (POINTER)partIn,
                partInLen);
        context->bufferLen += partInLen;
        *partOutLen = 0;
        return;
    }

    /* Fill the buffer and encrypt. */
    memcpy ((POINTER)(context->buffer + context->bufferLen), (POINTER)partIn,
            tempLen);
    DES_CBCUpdate (&context->cipherContext.des, partOut, partIn, 8);
    partIn += tempLen;
    partInLen -= tempLen;
    partOut += 8;
    *partOutLen = 8;

    /* Encrypt as many 8-byte blocks as possible. */
    tempLen = 8 * (partInLen / 8);
    DES_CBCUpdate (&context->cipherContext.des, partOut, partIn, tempLen);
    partIn += tempLen;
    partInLen -= tempLen;
    *partOutLen += tempLen;

    /* Length is now less than 8, so copy remainder to buffer. */
    memcpy ((POINTER)context->buffer, (POINTER)partIn,
            context->bufferLen = partInLen);
}

static void txt2data(unsigned char *text, unsigned char *data)
{
    while (*text != '\0')
    {
        /* High part of byte */
        if (*text >= '0' && *text <= '9')
            *data = (*text-'0') << 4;
        else if (*text >= 'a' && *text <= 'f')
            *data = (*text-'a'+10) << 4;
        else if (*text >= 'A' && *text <= 'F')
            *data = (*text-'A'+10) << 4;
        else
        {
            printf("Error! Not a hex character: %c (#%d)\n",*text,*text);
            exit(1);
        }
        text++;

        /* Low part of byte */
        if (*text >= '0' && *text <= '9')
            *data += *text-'0';
        else if (*text >= 'a' && *text <= 'f')
            *data += *text-'a'+10;
        else if (*text >= 'A' && *text <= 'F')
            *data += *text-'A'+10;
        else
        {
            printf("Error! Not a hex character: %c (#%d)\n",*text,*text);
            exit(1);
        }

        text++; data++;
    }
}

static unsigned long calccrc(void *ptr, int size)
{
    unsigned long crc;
    char *data;

    data = (char *) ptr;
    crc = -1UL;
    while (size > 0)
    {
        crc = (crc >> 8) ^ crc32tab [(int) ((crc ^ *data) & 0xffUL)];
        size--;
        data++;
    }
    return crc;
}

unsigned long serialnum;

int checkkeyfile(char *regname)
{
    R_ENVELOPE_CTX context;
    R_RANDOM_STRUCT randomStruct;
    unsigned char encryptedKey[MAX_ENCRYPTED_KEY_LEN];
    unsigned int outlen, datalen, encryptedKeyLen;
    FILE *Fkey;

    unsigned char data[100],crypted[500],*strp;

    regname[0] = '\0';

    /* Open key file */
#ifdef __linux__
    Fkey = FileOpen("~/.skyreader/skyread.key","rb");
#else
    Fkey = fopen("skyread.key","rb");
#endif
    if (Fkey == NULL) return 0;
    datalen = fread(data, 1, 1000, Fkey);
    fclose(Fkey);

    data[datalen] = '\0';

    strp = data;
    while (*strp != '\0')
    {
        if (*strp == '\r' || *strp == '\n')
        {
            *strp = '\0';
            break;
        }
        strp++;
    }

    txt2data(data,crypted); datalen /= 2;

    /* Read key file */
    InitRandomStruct (&randomStruct);
    if (!SealInit(&context, encryptedKey, &encryptedKeyLen, &PUBLIC_KEY1, &randomStruct))
        return 0;

    SealUpdate(&context, data, &outlen, crypted, datalen);

    /* Check CRC */
    memcpy(&serialnum, &data[0], 4);
    if (calccrc(data+4,strlen((char *) data+8)+4+1) != serialnum || strlen((char *) data+8) < 8)
    {
        printf("CRC error in key file! Please check if you wrote it wrong. If this message\n");
        printf("just keeps showing up, please ask a new registeration key from author\n");
        exit(99);
    }

    /* Key file ok! */
    memcpy(&serialnum, &data[4], 4);
    sprintf(regname, "%s, #%lu", data+8, serialnum);

    return 1;
}
